#include <Aria.h>

class GetRawEncoderPoseThread : public ArASyncTask
{
	ArCondition myCondition;
	ArMutex myMutex;
	ArRobot* myRobot;

public:
	GetRawEncoderPoseThread()
	{
		myCondition.setLogName("GetRawEncoderPoseThread");
	}

	void* runThread(void*)
	{
		ArPose pose;

		// Run until the thread is requested to end by another thread.
		while (this->getRunningWithLock())
		{
			myMutex.lock();

			myRobot->lock();
			pose = myRobot->getRawEncoderPose();
			myRobot->unlock();

			ArLog::log(ArLog::Normal, "GetRawEncoderPose thread: %8.4f", pose.getTh());

			// Unlock, then sleep.  We unlock before the sleep, so that while
			// we are sleeping, other threads that try to lock the mutex won't
			// be blocked until this thread is done sleeping.
			myMutex.unlock();
			ArUtil::sleep(1000);
		}

		ArLog::log(ArLog::Normal, "GetRawEncoderPose thread: requested stop running, ending thread.");
		return NULL;
	}

	/* Other threads can call this to wait for a condition eventually
	* signalled by this thread. (So note that in this example program, this
	* function is not executed within "Example thread", but is executed in the main thread.)
	*/
	void waitOnCondition()
	{
		myCondition.wait();
		ArLog::log(ArLog::Normal, " %s ArCondition object was signalled, done waiting for it.", myCondition.getLogName());
	}

	/* Get the counter. Not threadsafe, you must lock the mutex during access. */
	ArRobot* getRobot() { return myRobot; }

	/* Set the countner. Not threadsafe, you must lock the mutex during access. */
	void setRobot(ArRobot* robot) { myRobot = robot; }

	/* Lock the mutex object.  */
	void lockMutex() { myMutex.lock(); }

	/* Unlock the mutex object. */
	void unlockMutex() { myMutex.unlock(); }

};

int main(int argc, char** argv)
{
	// This object parses program options from the command line
	ArArgumentParser parser(&argc, argv);

	// Load some default values for command line arguments from /etc/Aria.args
	// (Linux) or the ARIAARGS environment variable.
	parser.loadDefaultArguments();

	// Central object that is an interface to the robot and its integrated
	// devices, and which manages control of the robot by the rest of the program.
	//a- Definition du robot
	printf("Robot definition \n");
	ArRobot robot;

	// Object that connects to the robot or simulator using program options
	ArRobotConnector robotConnector(&parser, &robot);

	// Connect to the robot, get some initial data from it such as type and name,
	// and then load parameter files for this robot.
	if (!robotConnector.connectRobot())
	{
		// Error connecting:
		// if the user gave the -help argumentp, then just print out what happened,
		// and continue so options can be displayed later.
		if (!parser.checkHelpAndWarnUnparsed())
		{
			ArLog::log(ArLog::Terse, "Could not connect to robot, will not have parameter file so options displayed later may not include everything");
		}
		// otherwise abort
		else
		{
			ArLog::log(ArLog::Terse, "Error, could not connect to robot.");
			Aria::logOptions();
			Aria::exit(1);
		}
	}

	if (!robot.isConnected())
	{
		ArLog::log(ArLog::Terse, "Internal error: robot connector succeeded but ArRobot::isConnected() is false!");
	}

	// Parse the command line options. Fail and print the help message if the parsing fails
	// or if the help was requested with the -help option
	if (!Aria::parseArgs() || !parser.checkHelpAndWarnUnparsed())
	{
		Aria::logOptions();
		Aria::exit(1);
		return 1;
	}
	//b- Initialisation du contexte
	printf("Aria Initialisation \n");
	Aria::init();

	//c- Definition capteur : Sonar
	printf("Sonar sensor definition \n");
	ArSonarDevice sonar;
	
	//d- Definition des actions du robot
	printf("Actions definition \n");
	//ArActionGoto action1("goto", 500, 350);  //permet d'envoyer le robot  une pose dfinie
	//ArActionTurn action2("turn", 400, 10);	//action de tourner le robot  une vitesse limite
	ArActionConstantVelocity action_constantVelocity;
	ArActionAvoidFront action_avoidFront;

	//   //Definition de mode !! non demand
	//   // ArModeWander wander(&robot, "wander", 'w', 'W');

	//e- Ajout du capteur sur le robot
	printf("Sonar added to robot \n");
	robot.addRangeDevice(&sonar);

	//f- Activation des moteurs et capteurs
	robot.comInt(ArCommands::ENABLE, 1);
	robot.comInt(ArCommands::SONAR, 1);
	robot.comInt(ArCommands::ENCODER, 2);

	robot.comStr(ArCommands::SAY, "\10\255\10\127\10\255\10\127\10\255\10\127\10\255\10\127");

	//g- Ajout d'actions du robot avec les priorits dfinies
	robot.addAction(&action_constantVelocity, 1);		//priorit de 1
	robot.addAction(&action_avoidFront, 2);		//priorit de 2

	GetRawEncoderPoseThread thread;
	thread.setRobot(&robot);

	ArLog::log(ArLog::Normal, "Main thread: Running new example thread ...");
	thread.runAsync();

	//h- Excution des comportements
	robot.run(true); // to return if it loses connection after which we exit the program.
	// Start the robot task loop running in a new background thread. The 'true' argument means if it loses
	// connection the task loop stops and the thread exits.
	//robot.runAsync(true);

	/* Loop, reading the value contained in the ExampleThread object.
	* We will also use ArUtil::sleep() to make this thread sleep each iteration,
	* instead of running as fast as possible and potentially preventing other
	* threads from access to the mutex and the shared counter.
	* When the counter reaches 10, break out of the loop and then wait on the
	* condition variable.
	*/

	// i- Attendre la fin de l'execution
	// Block execution of the main thread here and wait for the robot's task loop
	// thread to exit (e.g. by robot disconnecting, escape key pressed, or OS
	// signal)
	robot.waitForRunExit();

	// j- Fin du programme
	Aria::exit(0);

	return 0;
}